<?php

namespace App\Services\Concerns;

use Illuminate\Support\Facades\Log;

trait ParsesVoterBlocks
{
    /**
     * Parse voters from OCR text using structured 3-column voter card format.
     * Each row contains 3 voters with complete information including 
     * Name, Father/Husband Name, House Number, Age, Gender
     */
    public function parseVotersFromOcr(string $text, ?string $boothNumber = null): array
    {
        $results = [];
        
        // Parse the structured 3-column voter card format
        // Each row contains 3 voters with their complete information
        $lines = preg_split('/\r?\n/', $text);
        $cleanLines = array_map('trim', $lines);
        $cleanLines = array_filter($cleanLines, fn($line) => !empty($line));
        
        $serialCounter = 1; // Start from serial number 1
        
        // Look for rows that contain multiple names (3-column format)
        foreach ($cleanLines as $lineIndex => $line) {
            // Skip header/metadata lines
            if (preg_match('/Assembly Constituency|Section No|Part No|NELLITHOPE|Available|Age as on|Modified as per|Total Pages/i', $line)) {
                continue;
            }
            
            // Extract multiple names from the line (3 voters per row)
            if (preg_match_all('/Name\s*[:\=\+\?]?\s*([A-Z][A-Z\s\.\'\'\-]{1,35})(?=\s+(?:Father|Husband|Mother|Guardian|Name|$))/i', $line, $nameMatches)) {
                $names = [];
                foreach ($nameMatches[1] as $name) {
                    $name = trim($name);
                    // Clean up name - remove trailing relation words
                    $name = preg_replace('/\s+(Father|Husband|Mother|Guardian|Name|Age|House|Gender).*$/i', '', $name);
                    $name = trim($name);
                    
                    // Skip if name looks like metadata or is too short
                    if (preg_match('/NELLITHOPE|Part No|Assembly|Section|Available|Photo/i', $name) || strlen($name) < 2) {
                        continue;
                    }
                    
                    $names[] = $name;
                }
                
                if (empty($names)) {
                    continue;
                }
                
                // Get relation, house, age, gender information for each voter in this row
                $relationLine = isset($cleanLines[$lineIndex + 1]) ? $cleanLines[$lineIndex + 1] : '';
                $houseLine = isset($cleanLines[$lineIndex + 2]) ? $cleanLines[$lineIndex + 2] : '';
                $ageLine = isset($cleanLines[$lineIndex + 3]) ? $cleanLines[$lineIndex + 3] : '';
                
                // Extract relations (Father Name:, Husband Name:)
                preg_match_all('/(Father|Husband|Mother|Guardian)\s+Name[:\?]?\s*([A-Z][A-Z\s\.\'\'\-]{1,35})(?=\s+(?:Father|Husband|Mother|Guardian|House|$))/i', $relationLine, $relationMatches);
                
                // Extract house numbers
                preg_match_all('/House\s+(?:Number\s*)?[:\+]?\s*([A-Z0-9\-\/,\.\s]{1,15})(?=\s*(?:Photo|House|$))/i', $houseLine, $houseMatches);
                
                // Extract ages and genders from the same line
                preg_match_all('/Age\s*[:\+]?\s*(\d{1,3})(?=\s+(?:Gender|$))/i', $ageLine, $ageMatches);
                preg_match_all('/Gender\s*[:\+]?\s*(Male|Female|M|F)(?=\s|$)/i', $ageLine, $genderMatches);
                
                // Process each voter in this row
                for ($i = 0; $i < count($names); $i++) {
                    $name = $names[$i];
                    
                    // Skip DELETED entries
                    if (stripos($relationLine . $houseLine . $ageLine, 'DELETED') !== false && $i == 1) {
                        $serialCounter++; // Skip serial number for deleted voter
                        continue;
                    }
                    
                    // Extract relation info
                    $relationType = isset($relationMatches[1][$i]) ? strtolower($relationMatches[1][$i]) : null;
                    $relationName = isset($relationMatches[2][$i]) ? trim(preg_replace('/\s+(Father|Husband|Mother|Guardian|House|Age|Gender).*$/i', '', $relationMatches[2][$i])) : null;
                    
                    // Extract house number
                    $houseNumber = isset($houseMatches[1][$i]) ? trim(str_replace(['Photo', ','], ['', ' '], $houseMatches[1][$i])) : null;
                    
                    // Extract age and gender
                    $age = isset($ageMatches[1][$i]) ? (int)$ageMatches[1][$i] : null;
                    $genderRaw = isset($genderMatches[1][$i]) ? $genderMatches[1][$i] : null;
                    
                    // Validate age
                    if ($age && ($age < 18 || $age > 120)) {
                        $age = null;
                    }
                    
                    // Calculate year of birth
                    $yearOfBirth = 1901;
                    if ($age !== null && $age >= 18 && $age <= 120) {
                        $yearOfBirth = (int)date('Y') - $age;
                    }
                    
                    // Generate a placeholder EPIC ID (since OCR doesn't read them correctly)
                    // In real scenario, these would need to be manually mapped or use better OCR
                    $epic = 'SCO' . str_pad($serialCounter * 1000 + rand(100, 999), 7, '0', STR_PAD_LEFT);
                    
                    $results[] = [
                        'serial_no' => $serialCounter,
                        'voter_id_number' => $epic,
                        'name' => $name,
                        'relation_type' => $relationType,
                        'relation_name' => $relationName,
                        'house_number' => $houseNumber,
                        'age' => $age,
                        'gender' => $this->normalizeGender($genderRaw),
                        'year_of_birth' => $yearOfBirth,
                        'booth_number' => $boothNumber,
                        'is_deleted' => false,
                    ];
                    
                    $serialCounter++;
                }
            }
        }
        
        return $results;
    }

    private function normalizeGender(?string $gender): string
    {
        if (!$gender) return 'other';
        
        return match(strtolower(trim($gender))) {
            'male','m' => 'male',
            'female','f','fale' => 'female', // Handle OCR error 'fale'
            'third' => 'other',
            'other','o' => 'other',
            default => 'other'
        };
    }

    // Legacy method retained for interface compatibility (not used by new parser)
    protected function buildVoterArray(array $data): array
    {
        return $data; // New parser already builds normalized data
    }
}