<?php

require __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$app->make(Illuminate\Contracts\Console\Kernel::class)->bootstrap();

$voters = App\Models\Voter::where('booth_id', 54)->orderBy('serial_number')->get(['serial_number', 'voter_id_number', 'name', 'house_number', 'age', 'gender']);

echo "=== VOTER DATA VERIFICATION ===" . PHP_EOL;
echo "Comparing imported data with original image..." . PHP_EOL;
echo PHP_EOL;

// Expected data from image (rows 1-30, excluding deleted #22)
$expected = [
    ['serial' => 1, 'id' => 'TMC0087361', 'name' => 'JAYEMKONDAAN'],
    ['serial' => 2, 'id' => 'TMC0087346', 'name' => 'Girija Shanker'],
    ['serial' => 3, 'id' => 'TMC0087338', 'name' => 'DOUCHEYANDHABHARATHI'],
    ['serial' => 4, 'id' => 'TMC0097204', 'name' => 'Padmini'],
    ['serial' => 5, 'id' => 'TMC0097196', 'name' => 'KALAVATHY'],
    ['serial' => 6, 'id' => 'TMC0097188', 'name' => 'Lalitha'],
    ['serial' => 7, 'id' => 'TMC0097170', 'name' => 'Sopithakumar'],
    ['serial' => 8, 'id' => 'TMC0097162', 'name' => 'SUGANTHA BALAJI'],
    ['serial' => 9, 'id' => 'GKY0718932', 'name' => 'DHEERKA SUBHA'],
    ['serial' => 10, 'id' => 'TMC0097147', 'name' => 'MALATHI'],
    ['serial' => 11, 'id' => 'CMS0274134', 'name' => 'SENTHILKUMARAN'],
    ['serial' => 12, 'id' => 'CMS0315440', 'name' => 'BARANI PRABU'],
    ['serial' => 13, 'id' => 'TMC0002519', 'name' => 'YOGESHKUMAR'],
    ['serial' => 14, 'id' => 'CMS0301267', 'name' => 'SUPRADHEEPA'],
    ['serial' => 15, 'id' => 'TMC0002436', 'name' => 'MARIA REISAL KATHREEN'],
    ['serial' => 16, 'id' => 'TMC0002972', 'name' => 'PREETHI PRABHA'],
    ['serial' => 17, 'id' => 'TMC0021535', 'name' => 'KOPIGA BHARATHY'],
    ['serial' => 18, 'id' => 'TMC0031799', 'name' => 'KUZHAL MALAN'],
    ['serial' => 19, 'id' => 'TMC0158048', 'name' => 'DHAYAAPARAN'],
    ['serial' => 20, 'id' => 'TMC0097121', 'name' => 'RATHINAM ALIAS MANOHAR'],
    ['serial' => 21, 'id' => 'HJN0167643', 'name' => 'KALAISELVI'],
    // 22 is DELETED - TMC0058552 SARANYA
    ['serial' => 23, 'id' => 'TMC0047233', 'name' => 'GOBISHANKAR ALIAS VIGNESH'],
    ['serial' => 24, 'id' => 'TMC0062570', 'name' => 'PRIYANGA'],
    ['serial' => 25, 'id' => 'TMC0180661', 'name' => 'VS PRAVEEN RAJAN'],
    ['serial' => 26, 'id' => 'TMC0097097', 'name' => 'Saraswathi'],
    ['serial' => 27, 'id' => 'GKY0441113', 'name' => 'USHA'],
    ['serial' => 28, 'id' => 'CMS0258822', 'name' => 'Sridevi'],
    ['serial' => 29, 'id' => 'CMS0294496', 'name' => 'Srinivasan'],
    ['serial' => 30, 'id' => 'TMC0065854', 'name' => 'PAVITHRA'],
];

$mismatches = [];
$correct = 0;

foreach ($voters as $voter) {
    $serial = $voter->serial_number;
    if ($serial === null) {
        echo "⚠️  Serial NULL: " . $voter->voter_id_number . " - " . $voter->name . PHP_EOL;
        continue;
    }
    
    $expectedRecord = array_filter($expected, fn($e) => $e['serial'] == $serial);
    if (empty($expectedRecord)) {
        $mismatches[] = ['serial' => $serial, 'imported_id' => $voter->voter_id_number, 'imported_name' => $voter->name, 'reason' => 'Serial not in expected list'];
        continue;
    }
    
    $expectedRecord = reset($expectedRecord);
    
    if ($voter->voter_id_number !== $expectedRecord['id']) {
        $mismatches[] = [
            'serial' => $serial,
            'expected_id' => $expectedRecord['id'],
            'imported_id' => $voter->voter_id_number,
            'expected_name' => $expectedRecord['name'],
            'imported_name' => $voter->name
        ];
    } else {
        $correct++;
    }
}

echo PHP_EOL;
echo "=== VERIFICATION RESULTS ===" . PHP_EOL;
echo "Total voters checked: " . count($voters) . PHP_EOL;
echo "Correct matches: " . $correct . PHP_EOL;
echo "Mismatches found: " . count($mismatches) . PHP_EOL;

if (!empty($mismatches)) {
    echo PHP_EOL;
    echo "❌ MISMATCHES DETECTED:" . PHP_EOL;
    foreach ($mismatches as $m) {
        echo "  Serial " . $m['serial'] . ":" . PHP_EOL;
        echo "    Expected: " . ($m['expected_id'] ?? 'N/A') . " - " . ($m['expected_name'] ?? 'N/A') . PHP_EOL;
        echo "    Imported: " . $m['imported_id'] . " - " . $m['imported_name'] . PHP_EOL;
        if (isset($m['reason'])) echo "    Reason: " . $m['reason'] . PHP_EOL;
    }
} else {
    echo PHP_EOL;
    echo "✅ All voter IDs match correctly!" . PHP_EOL;
}

// Additional check: verify deleted voter is not in database
echo PHP_EOL;
echo "=== DELETED VOTER CHECK ===" . PHP_EOL;
$deletedVoter = App\Models\Voter::where('voter_id_number', 'TMC0058552')->first();
if ($deletedVoter) {
    echo "❌ ERROR: Deleted voter (TMC0058552 - SARANYA) found in database!" . PHP_EOL;
} else {
    echo "✅ Deleted voter (Serial 22: TMC0058552 - SARANYA) correctly excluded" . PHP_EOL;
}

// Check for duplicate voter IDs
echo PHP_EOL;
echo "=== DUPLICATE VOTER ID CHECK ===" . PHP_EOL;
$allVoters = App\Models\Voter::where('booth_id', 54)->get(['voter_id_number']);
$voterIds = $allVoters->pluck('voter_id_number')->toArray();
$duplicates = array_filter(array_count_values($voterIds), fn($count) => $count > 1);

if (!empty($duplicates)) {
    echo "❌ DUPLICATE VOTER IDs FOUND:" . PHP_EOL;
    foreach ($duplicates as $voterId => $count) {
        echo "  $voterId appears $count times" . PHP_EOL;
    }
} else {
    echo "✅ No duplicate voter IDs found" . PHP_EOL;
}
