<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Artisan;
use App\Http\Controllers\AdminController;
use App\Http\Controllers\RationCardController;
use App\Models\Admin;
use Illuminate\Support\Facades\Hash;
use App\Http\Controllers\Api\ImageUploadController;
use App\Http\Controllers\BoothController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\GiftController;
use App\Http\Controllers\GrievanceController;
use App\Http\Controllers\SMSController;
use App\Http\Controllers\StreetController;
use App\Http\Controllers\SurveyController;
use App\Http\Controllers\VoterController;
use App\Http\Controllers\VoterImageImportController;
use App\Http\Controllers\VoterImageBatchController;
use App\Http\Controllers\PdfToPngController;
use App\Http\Controllers\SimplePdfUploadController;
use App\Http\Controllers\WhatsAppController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| is assigned the "api" middleware group. Enjoy building your API!
|
*/

// ==========================================
// UTILITY & TEST ROUTES
// ==========================================

Route::get('/test', function () {
    return 'Hello World';
});

Route::get('migrate', function () {
    Artisan::call('migrate:fresh');
});

Route::get('migrateLatest', function () {
    Artisan::call('migrate', [
        '--force' => true // To run in production without confirmation
    ]);
    return 'Migration completed successfully!';
});

// ==========================================
// ADMIN MANAGEMENT ROUTES
// ==========================================

Route::prefix('admin')->group(function () {
    Route::post('login',  [AdminController::class, 'loginAdminUser']);
    Route::post('create', [AdminController::class, 'createAdmin']);
    Route::post('change-password', [AdminController::class, 'changeAdminPassword']);
});

// ==========================================
// RATION CARD MANAGEMENT ROUTES
// ==========================================

Route::prefix('rationCard')->group(function () {
    Route::post('create',    [RationCardController::class, 'createRationCard']);
    Route::post('create-only', [RationCardController::class, 'createRationCardOnly']);
    Route::post('create-test', [RationCardController::class, 'createRationCardTest']); // Test endpoint without encryption
    Route::put('edit/{id}',  [RationCardController::class, 'editRationCard']);
    Route::put('edit-only/{id}', [RationCardController::class, 'editRationCardOnly']); // Update only ration card data (no voters)
    Route::get('get/{id}',   [RationCardController::class, 'getRationCard']);
    Route::get('getAll',     [RationCardController::class, 'getRationCards']);
    Route::POST('search',    [RationCardController::class, 'searchRationCardOrVoterId']);
    Route::POST('searchByAnyId',    [RationCardController::class, 'getRationCardAndVoterId']);
    Route::delete('delete/{rationCardNumber}', [RationCardController::class, 'deleteRationCard']);
    Route::post('generateCard/{id}', [RationCardController::class, 'generateCard']); // Generate ration card image

});


// ==========================================
// BOOTH MANAGEMENT ROUTES
// ==========================================

Route::prefix('booth')->group(function () {
    Route::post('create',                   [BoothController::class, 'createBooth']);
    Route::put('edit/{id}',                 [BoothController::class, 'editBooth']);
    Route::get('get/{id}',                  [BoothController::class, 'getBooth']);
    Route::get('getAll',                    [BoothController::class, 'getAllBooths']);
    Route::get('getAllWithLists',           [BoothController::class, 'getBoothsWithLists']);
    Route::get('getStaticList',              [BoothController::class, 'getStaticList']);
    Route::delete('delete/{booth_number}',  [BoothController::class, 'deleteBooth']);
});

// ==========================================
// STREET MANAGEMENT ROUTES
// ==========================================

Route::prefix('street')->group(function () {
    Route::post('create', [StreetController::class, 'create']);
    Route::get('getAll', [StreetController::class, 'getAllStreets']);
    Route::put('edit/{id}', [StreetController::class, 'editStreet']);
    Route::delete('delete/{id}', [StreetController::class, 'deleteStreet']);
});

// ==========================================
// VOTER MANAGEMENT ROUTES
// ==========================================

Route::prefix('voters')->group(function () {
    // Standalone voter management
    Route::post('create', [VoterController::class, 'createVoter']); // Original encrypted JSON endpoint
    Route::post('create-with-image', [VoterController::class, 'createVoterWithImage']); // New multipart endpoint
    Route::get('get/{voterId}', [VoterController::class, 'getVoter']);
    Route::put('update/{voterId}', [VoterController::class, 'updateVoter']);
    Route::delete('delete/{voterId}', [VoterController::class, 'deleteVoter']);
    
    // Bulk import functionality
    Route::post('import', [VoterController::class, 'importVoters']); // Import from CSV
    Route::post('import-excel', [VoterController::class, 'importVotersExcel']); // Import from Excel
    Route::get('download-template', [VoterController::class, 'downloadTemplate']); // Download CSV template
    Route::get('download-excel-template', [VoterController::class, 'downloadExcelTemplate']); // Download Excel template
    
    // Form data for voter creation
    Route::get('getFormData', [VoterController::class, 'getFormData']);
    Route::get('getBooths', [VoterController::class, 'getBooths']);
    Route::get('getStreets', [VoterController::class, 'getStreets']);
    
    // Voter to ration card mapping
    Route::post('map-to-ration-card', [VoterController::class, 'mapVoterToRationCard']); // Encrypted JSON endpoint
    Route::post('map-to-ration-card-form', [VoterController::class, 'mapVoterToRationCardForm']); // Multipart form endpoint
    
    // Search voter by voter ID number
    Route::get('search/{voterIdNumber}', [VoterController::class, 'searchVoterByIdNumber']);
    
    // Existing routes
    Route::get('getByStreetId/{streetId}', [VoterController::class, 'getByStreetId']);
    Route::get('getByBoothIdAndStreetId/{boothId}/{streetId}', [VoterController::class, 'getByBoothIdAndStreetId']);
    Route::get('getAll', [VoterController::class, 'getAllVoters']);
});

// ==========================================
// PDF TO PNG CONVERSION
// ==========================================

Route::prefix('pdf-to-png')->group(function () {
    Route::post('extract', [PdfToPngController::class, 'extractPdfToPng']); // params: pdf_file, reset (optional), dpi (optional) - Auto-extracts constituency & booth from PDF
});

// ==========================================
// IMAGE IMPORT ROUTES (OCR-based Voter Extraction)
// ==========================================

Route::prefix('image-import')->group(function () {
    // Dispatch OCR jobs for images already present in a directory
    Route::post('dispatch', VoterImageImportController::class); // params: directory, booth, limit
    // Batch import from hierarchical folder structure: Constituency/[ConstituencyName]/[BoothNumber]
    Route::post('run', [VoterImageBatchController::class, 'run']); // params: constituency, booth_number, start_index, max_images
    // Process all booths in a constituency
    Route::post('run-constituency', [VoterImageBatchController::class, 'runConstituency']); // params: constituency
    
    // Debug endpoint
    Route::post('debug', function (Request $request) {
        return response()->json([
            'all_input' => $request->all(),
            'constituency' => $request->input('constituency'),
            'booth_number' => $request->input('booth_number'),
            'has_constituency' => $request->has('constituency'),
            'content_type' => $request->header('Content-Type'),
            'raw_body' => $request->getContent(),
            'method' => $request->method(),
            'is_json' => $request->isJson()
        ]);
    });
});

// ==========================================
// EXCEL EXPORT OPERATIONS
// ==========================================

Route::prefix('excelOperation')->group(function () {
    // General exports (with optional filtering via POST request body)
    Route::post('downloadRationCards', [RationCardController::class, 'downloadAsExcel']);
    Route::post('downloadVoterIds',    [RationCardController::class, 'downloadVotersAsExcel']);
    Route::post('downloadBooths',       [BoothController::class, 'downloadAsExcel']);

    // Legacy GET routes (backward compatibility)
    Route::post('downloadRationCards',  [RationCardController::class, 'downloadAsExcel']);
    Route::post('downloadVoterIds',     [RationCardController::class, 'downloadVotersAsExcel']);

    // Booth-specific exports (legacy routes)
    Route::post('downloadRationCardsByBooth/{boothNumber}', [RationCardController::class, 'downloadAsExcelByBooth']);
    Route::post('downloadVoterIdsByBooth/{boothNumber}',    [RationCardController::class, 'downloadVotersAsExcelByBooth']);
    Route::post('downloadRationCardsAsExcelByBooth/{boothNumber}', [RationCardController::class, 'downloadRationCardsAsExcelByBooth']);

    // Unmapped voters export (voters without ration_card_id)
    Route::post('downloadUnmappedVotersByBooth/{boothNumber}', [RationCardController::class, 'downloadUnmappedVotersByBooth']);

    // Mapped voters export (voters with ration_card_id)
    Route::post('downloadMappedVotersByBooth/{boothNumber}', [RationCardController::class, 'downloadMappedVotersByBooth']);

    Route::post('downloadGifted/{giftId}',  [GiftController::class, 'downloadGiftDistributionExcel'])->defaults('gifted', true);
    Route::post('downloadPending/{giftId}', [GiftController::class, 'downloadGiftDistributionExcel'])->defaults('gifted', false);

});

// ==========================================
// IMAGE UPLOAD MANAGEMENT
// ==========================================

Route::prefix('imageHandler')->group(function () {
    // Ration card image handling
    Route::post('upload-image',   [ImageUploadController::class, 'uploadImage']);
    Route::delete('delete-image', [ImageUploadController::class, 'deleteImage']);
    
    // Voter image handling
    Route::post('upload-voter-image',   [ImageUploadController::class, 'uploadVoterImage']);
    Route::delete('delete-voter-image', [ImageUploadController::class, 'deleteVoterImage']);
});

// ==========================================
// GIFT MANAGEMENT ROUTES
// ==========================================

Route::prefix('gifts')->group(function () {
    Route::post('create',           [GiftController::class, 'create']);
    Route::get('get/{id}',          [GiftController::class, 'getById']);
    Route::get('getAll',            [GiftController::class, 'getAll']);
    Route::put('update/{id}',       [GiftController::class, 'update']);
    Route::delete('delete/{id}',    [GiftController::class, 'delete']);
    Route::get('getDeletedAll',     [GiftController::class, 'getAllDeleted']);

    // Gift distribution tracking
    Route::post('distribute',       [GiftController::class, 'distributeGift']);
    Route::post('distribute/single', [GiftController::class, 'distributeGiftBySingle']);
    Route::post('revert',            [GiftController::class, 'revertGiftDistribution']);
    
    // Get gifts distributed to a specific ration card
    Route::get('ration-card/{rationCardNumber}', [GiftController::class, 'getGiftsByRationCard']);
    
    // Excel export routes for gift distribution
  });

// Alias endpoint for gift distribution revert
Route::post('giftsRevert', [GiftController::class, 'revertGiftDistribution']);

// ==========================================
// SMS MANAGEMENT ROUTES
// ==========================================

Route::prefix('sms')->group(function () {
    // Type 4: Custom mobile numbers (existing functionality - updated)
    Route::post('send',         [SMSController::class, 'sendCustomSMS']);
    
    // Type 1: Send to all head voters
    Route::post('sendToAll',    [SMSController::class, 'sendToAll']);
    
    // Type 2: Send to gifted ration card holders
    Route::post('sendToGifted', [SMSController::class, 'sendToGifted']);
    
    // Type 3: Send to pending (non-gifted) ration card holders
    Route::post('sendToPending', [SMSController::class, 'sendToPending']);
    
});

// ==========================================
// SELF SURVEY MANAGEMENT ROUTES
// ==========================================

Route::prefix('survey')->group(function () {
    Route::post('create',                       [SurveyController::class, 'createSurvey']);
    Route::get('getAll',                        [SurveyController::class, 'getAllSurveys']);
    Route::get('get/{id}',                      [SurveyController::class, 'getSurveyById']);
    Route::get('{surveyId}/getAllBooths',       [SurveyController::class, 'getAllBooths']);
    Route::post('{surveyId}/voter/rating',      [SurveyController::class, 'addOrUpdateVoterRating']);
    Route::delete('delete/{id}',                [SurveyController::class, 'deleteSurvey']);
});

// ==========================================
// GRIEVANCE MANAGEMENT ROUTES
// ==========================================

Route::prefix('grievance')->group(function () {
    Route::post('create',                       [GrievanceController::class, 'create']);
    Route::put('update/{id}',                   [GrievanceController::class, 'update']);
    Route::get('get/{id}',                      [GrievanceController::class, 'getById']);
    Route::get('getAll',                        [GrievanceController::class, 'getAll']);
    Route::get('getByStatus/{status}',          [GrievanceController::class, 'getByStatus']);
    Route::get('getByVoterId/{voterId}',        [GrievanceController::class, 'getByVoterId']);
    Route::get('getByRationCard/{rationCardNumber}', [GrievanceController::class, 'getByRationCardNumber']);
    Route::delete('delete/{id}',                [GrievanceController::class, 'delete']);
    Route::get('statuses',                      [GrievanceController::class, 'getStatuses']);
    
    // Pagination routes
    Route::get('paginated',                     [GrievanceController::class, 'getPaginatedGrievances']);
    Route::get('paginated-all',                 [GrievanceController::class, 'getAllPaginated']);
    
    // Debug route
    Route::get('debug-search/{rationCardNumber}', function($rationCardNumber) {
        $grievances = \App\Models\Grievance::with(['voter.rationCard', 'rationCard'])
            ->where(function ($q) use ($rationCardNumber) {
                $q->where('ration_card_number', $rationCardNumber)
                  ->orWhereHas('voter', function ($voterQuery) use ($rationCardNumber) {
                      $voterQuery->whereHas('rationCard', function ($rcQuery) use ($rationCardNumber) {
                          $rcQuery->where('ration_card_number', $rationCardNumber);
                      });
                  });
            })->get();
            
        return response()->json([
            'search_for' => $rationCardNumber,
            'found_grievances' => $grievances->count(),
            'grievances' => $grievances->map(function($g) {
                return [
                    'id' => $g->id,
                    'voter_id' => $g->voter_id,
                    'ration_card_number' => $g->ration_card_number,
                    'title' => $g->title,
                    'voter' => $g->voter ? [
                        'voter_id_number' => $g->voter->voter_id_number,
                        'name' => $g->voter->name,
                        'ration_card_id' => $g->voter->ration_card_id,
                        'ration_card' => $g->voter->rationCard ? [
                            'ration_card_number' => $g->voter->rationCard->ration_card_number
                        ] : null
                    ] : null,
                    'direct_ration_card' => $g->rationCard ? [
                        'ration_card_number' => $g->rationCard->ration_card_number
                    ] : null
                ];
            })
        ]);
    });
});



// ==========================================
// DASHBOARD & ANALYTICS
// ==========================================

Route::get('getDashBoardData', [DashboardController::class, 'getDashboardCounts']);
Route::get('getAllBoothsWithVoters', [DashboardController::class, 'getAllBoothsWithVoters']);
Route::post('downloadTotalUnmappedVotersAsExcel', [DashboardController::class, 'downloadTotalUnmappedVotersAsExcel']);



// ==========================================
// DEVELOPMENT & TESTING UTILITIES
// ==========================================

Route::post('test-encryption', function (Request $request) {
    try {
        $rawData = $request->getContent();
        
        return response()->json([
            'status' => 'debug',
            'raw_data_length' => strlen($rawData),
            'raw_data_preview' => substr($rawData, 0, 100) . '...',
            'content_type' => $request->header('Content-Type'),
            'method' => $request->method(),
            'headers' => $request->headers->all()
        ]);
        
    } catch (\Exception $e) {
        return response()->json([
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
    }
});

Route::post('test-decryption', function (Request $request) {
    try {
        $data = $request->getContent();
        
        if (empty($data)) {
            return response()->json(['error' => 'No data received']);
        }
        
        // Test decryption
        $decryptedData = \App\Security\DecryptUtils::decryptRequestData($data);
        
        return response()->json([
            'status' => 'success',
            'encrypted_length' => strlen($data),
            'decrypted_data' => $decryptedData,
            'decrypted_type' => gettype($decryptedData),
            'is_array' => is_array($decryptedData)
        ]);
        
    } catch (\Exception $e) {
        return response()->json([
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
    }
});

Route::get('debug-grievances/{rationCardNumber}', function ($rationCardNumber) {
    try {
        $grievances = \App\Models\Grievance::all();
        $totalCount = $grievances->count();
        
        // Check direct matches
        $directMatches = \App\Models\Grievance::where('ration_card_number', $rationCardNumber)->get();
        
        // Check voter relationship matches
        $voterMatches = \App\Models\Grievance::whereHas('voter', function ($query) use ($rationCardNumber) {
            $query->where('ration_card_number', $rationCardNumber);
        })->get();
        
        // Check rationCard relationship matches
        $rationCardMatches = \App\Models\Grievance::whereHas('rationCard', function ($query) use ($rationCardNumber) {
            $query->where('ration_card_number', $rationCardNumber);
        })->get();
        
        // Sample grievance data
        $sampleGrievances = \App\Models\Grievance::limit(5)->get()->map(function ($g) {
            return [
                'id' => $g->id,
                'ration_card_number' => $g->ration_card_number,
                'voter_id' => $g->voter_id,
                'title' => $g->title,
                'has_voter' => $g->voter ? true : false,
                'has_ration_card' => $g->rationCard ? true : false,
                'voter_ration_card' => $g->voter ? $g->voter->ration_card_number : null,
                'relation_ration_card' => $g->rationCard ? $g->rationCard->ration_card_number : null
            ];
        });
        
        return response()->json([
            'search_for' => $rationCardNumber,
            'total_grievances' => $totalCount,
            'direct_matches' => $directMatches->count(),
            'voter_matches' => $voterMatches->count(),
            'ration_card_matches' => $rationCardMatches->count(),
            'sample_grievances' => $sampleGrievances,
            'direct_match_details' => $directMatches->pluck('ration_card_number'),
            'voter_match_details' => $voterMatches->load('voter')->map(function($g) {
                return ['grievance_id' => $g->id, 'voter_ration_card' => $g->voter ? $g->voter->ration_card_number : null];
            }),
            'ration_card_match_details' => $rationCardMatches->load('rationCard')->map(function($g) {
                return ['grievance_id' => $g->id, 'ration_card_number' => $g->rationCard ? $g->rationCard->ration_card_number : null];
            })
        ]);
        
    } catch (\Exception $e) {
        return response()->json([
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
    }
});

Route::get('create-sample-grievances', function () {
    $sampleGrievances = [
        [
            'voter_id' => 'V001',
            'ration_card_number' => 'RC123456',
            'title' => 'Ration card not received',
            'description' => 'Applied for ration card 3 months ago but still not received',
            'status' => 'OPEN',
            'reported_at' => now()->subDays(5),
            'target_resolve_by' => now()->addDays(10),
        ],
        [
            'voter_id' => 'V002', 
            'ration_card_number' => 'RC123456',
            'title' => 'Incorrect name on ration card',
            'description' => 'Name spelling is wrong on the ration card',
            'status' => 'IN_PROGRESS',
            'reported_at' => now()->subDays(3),
            'target_resolve_by' => now()->addDays(7),
        ],
        [
            'voter_id' => 'V003',
            'ration_card_number' => 'RC789012',
            'title' => 'Missing family member',
            'description' => 'Newly married daughter-in-law not added to ration card',
            'status' => 'OPEN',
            'reported_at' => now()->subDays(2),
            'target_resolve_by' => now()->addDays(15),
        ],
        [
            'voter_id' => 'V004',
            'ration_card_number' => 'RC456789',
            'title' => 'Ration shop quality issue',
            'description' => 'Poor quality rice being distributed',
            'status' => 'CLOSED',
            'reported_at' => now()->subDays(20),
            'target_resolve_by' => now()->subDays(5),
            'closed_at' => now()->subDays(2),
        ],
        [
            'voter_id' => 'V005',
            'ration_card_number' => 'RC123456',
            'title' => 'Ration shop timing issue',
            'description' => 'Shop opens very late, causing problems for working people',
            'status' => 'HOLD',
            'reported_at' => now()->subDays(1),
            'target_resolve_by' => now()->addDays(5),
        ],
        [
            'voter_id' => 'V006',
            'ration_card_number' => 'RC999888',
            'title' => 'Duplicate ration card issue',
            'description' => 'Two ration cards issued for same family',
            'status' => 'UNRESOLVED',
            'reported_at' => now()->subDays(15),
            'target_resolve_by' => now()->addDays(20),
        ],
    ];

    $created = 0;
    foreach ($sampleGrievances as $grievanceData) {
        try {
            \App\Models\Grievance::create($grievanceData);
            $created++;
        } catch (Exception $e) {
            // Skip if already exists or error
        }
    }

    return response()->json([
        'status' => 'success',
        'message' => "Created $created sample grievances successfully",
        'total_created' => $created,
        'sample_ration_cards' => ['RC123456', 'RC789012', 'RC456789', 'RC999888'],
        'test_urls' => [
            'search_by_ration_card' => url('/api/grievance/paginated?ration_card_number=RC123456&per_page=5'),
            'all_open_grievances' => url('/api/grievance/paginated-all?status=OPEN&per_page=10'),
            'all_grievances' => url('/api/grievance/paginated-all?per_page=15')
        ]
    ]);
});

// WhatsApp API routes
Route::prefix('whatsapp')->group(function () {
    Route::post('send', [WhatsAppController::class, 'sendSimpleMessage']);
    Route::get('test-config', [WhatsAppController::class, 'testConfiguration']);
    Route::post('debug', [WhatsAppController::class, 'debugRequest']);
    Route::post('send-template', [WhatsAppController::class, 'sendTemplateTest']);
    Route::post('bulk-template', [WhatsAppController::class, 'sendBulkTemplate']);

});


Route::get('create-test-admin', function () {
    $username = 'guest';
    $plainPassword = '123456';

    // Check if admin already exists
    $existing = Admin::where('username', $username)->first();
    if ($existing) {
        return response()->json([
            'status' => 'exists',
            'message' => 'Test admin already exists',
            'admin_id' => $existing->id
        ]);
    }

    // Create new test admin
    $admin = Admin::create([
        'username' => $username,
        'password' => $plainPassword,
        'role' => 'guest'
    ]);

    return response()->json([
        'status' => 'created',
        'message' => 'Test admin created successfully',
        'admin_id' => $admin->id,
        'credentials' => [
            'username' => $username,
            'password' => $plainPassword
        ]
    ]);
});
