<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class GiftedRationCard extends Model
{
    protected $table = 'gifted_ration_cards';
    
    protected $fillable = [
        'gift_id',
        'ration_card_number',
        'is_deleted'
    ];

    protected $casts = [
        'is_deleted' => 'boolean'
    ];

    /**
     * Relationship with Gift model
     */
    public function gift()
    {
        return $this->belongsTo(Gift::class, 'gift_id', 'id');
    }

    /**
     * Relationship with RationCard model
     */
    public function rationCard()
    {
        return $this->belongsTo(RationCard::class, 'ration_card_number', 'ration_card_number');
    }

    /**
     * Scope to get only non-deleted records
     */
    public function scopeActive($query)
    {
        return $query->where('is_deleted', false);
    }

    /**
     * Scope to get deleted records
     */
    public function scopeDeleted($query)
    {
        return $query->where('is_deleted', true);
    }

    /**
     * Check if a gift has already been distributed to a ration card
     */
    public static function isAlreadyDistributed($giftId, $rationCardNumber)
    {
        return self::where('gift_id', $giftId)
                   ->where('ration_card_number', $rationCardNumber)
                   ->where('is_deleted', false)
                   ->exists();
    }

    /**
     * Get all ration cards that received a specific gift
     */
    public static function getDistributedRationCards($giftId)
    {
        return self::where('gift_id', $giftId)
                   ->where('is_deleted', false)
                   ->pluck('ration_card_number')
                   ->toArray();
    }
}