<?php

namespace App\Http\Controllers;

use App\Models\Voter;
use App\Services\VoterService;
use App\Security\DecryptUtils;
use App\ResponseHandler\Response;
use App\Validations\Validation;
use App\Constants\Constants;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class VoterController extends Controller
{
    protected $voterService;
    protected $response;

    public function __construct(VoterService $voterService, Response $response)
    {
        $this->voterService = $voterService;
        $this->response = $response;
    }

    /**
     * Create a standalone voter
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function createVoter(Request $request)
    {
        $data = $request->getContent();

        if (empty($data)) {
            return $this->response->respondWithError('Request data is empty');
        }

        // Step 2: Decrypt the incoming data
        $decryptedData = DecryptUtils::decryptRequestData($data);

        // Step 3: Ensure it's a valid array
        if ($decryptedData == null || !is_array($decryptedData)) {
            return $this->response->respondWithError('Invalid or empty request data');
        }

        // Step 4: Validation
        $validation = Validation::validationForVoter($decryptedData);
        if ($validation->getStatusCode() !== Constants::SUCCESS_CODE) {
            return $validation; // return validation error
        }

        // Step 5: Pass to service
        return $this->voterService->createVoter($decryptedData);
    }

    /**
     * Get voter by ID
     * 
     * @param int $voterId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getVoter($voterId)
    {
        return $this->voterService->getVoterById($voterId);
    }

    /**
     * Update voter
     * 
     * @param Request $request
     * @param int $voterId
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateVoter(Request $request, $voterId)
    {
        $data = $request->getContent();

        if (empty($data)) {
            return $this->response->respondWithError('Request data is empty');
        }

        // Decrypt the incoming data
        $decryptedData = DecryptUtils::decryptRequestData($data);

        if ($decryptedData == null || !is_array($decryptedData)) {
            return $this->response->respondWithError('Invalid or empty request data');
        }

        // Validation
        $validation = Validation::validationForVoter($decryptedData);
        if ($validation->getStatusCode() !== Constants::SUCCESS_CODE) {
            return $validation;
        }

        // Pass to service
        return $this->voterService->updateVoter($voterId, $decryptedData);
    }

    /**
     * Delete voter
     * 
     * @param int $voterId
     * @return \Illuminate\Http\JsonResponse
     */
    public function deleteVoter($voterId)
    {
        if (empty($voterId)) {
            return $this->response->respondWithError('Voter ID is required');
        }

        return $this->voterService->deleteVoter($voterId);
    }

    /**
     * Get booths for dropdown
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function getBooths()
    {
        return $this->voterService->getBoothsForDropdown();
    }

    /**
     * Get streets for dropdown
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function getStreets()
    {
        return $this->voterService->getStreetsForDropdown();
    }

    /**
     * Get all form data for voter creation (booths, streets, genders)
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function getFormData()
    {
        return $this->voterService->getVoterFormData();
    }
    /**
     * Get all voters by street ID
     * 
     * @param int $streetId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getByStreetId($streetId)
    {
        try {
            $voters = Voter::where('street_id', $streetId)
                ->where('is_deleted', false)
                ->with(['rationCard', 'street', 'booth'])
                ->get()
                ->map(function ($voter) {
                    return $this->formatVoterData($voter);
                });

            return response()->json([
                'status' => 200,
                'message' => 'Voters retrieved successfully',
                'data' => $voters,
                'total' => $voters->count(),
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Failed to retrieve voters',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get all voters
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAllVoters()
    {
        try {
            $voters = Voter::where('is_deleted', false)
                ->with(['rationCard', 'street', 'booth'])
                ->get()
                ->map(function ($voter) {
                    return $this->formatVoterData($voter);
                });

            return response()->json([
                'status' => 200,
                'message' => 'All voters retrieved successfully',
                'data' => $voters,
                'total' => $voters->count(),
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Failed to retrieve voters',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get all voters by booth ID and street ID
     * 
     * @param string $boothId
     * @param int $streetId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getByBoothIdAndStreetId($boothId, $streetId)
    {
        try {
            $voters = Voter::where('booth_id', $boothId)
                ->where('street_id', $streetId)
                ->where('is_deleted', false)
                ->with(['rationCard', 'street', 'booth'])
                ->get()
                ->map(function ($voter) {
                    return $this->formatVoterData($voter);
                });

            return response()->json([
                'status' => 200,
                'message' => 'Voters retrieved successfully',
                'data' => $voters,
                'total' => $voters->count(),
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Failed to retrieve voters',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Create voter with image upload (multipart/form-data)
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function createVoterWithImage(Request $request)
    {
        try {
            // Step 1: Validate the form data
            $request->validate([
                'voter_id_number' => 'required|string|max:50',
                'name' => 'required|string|max:255',
                'gender' => 'required|in:Male,Female,Other',
                'year_of_birth' => 'nullable|integer|min:1900|max:' . date('Y'),
                'date_of_birth' => 'nullable|date_format:Y-m-d',
                'anniversary_date' => 'nullable|date_format:Y-m-d',
                'is_head' => 'nullable|boolean',
                'booth_id' => 'nullable|integer',
                'booth_number' => 'nullable|string|max:50',
                'mobile_number' => 'nullable|regex:/^[0-9]{10}$/',
                'aadhar_number' => 'nullable|regex:/^[0-9]{12}$/',
                'street_id' => 'nullable|integer',
                'street_name' => 'nullable|string|max:255',
                'ward_no' => 'nullable|string|max:50',
                'caste' => 'nullable|string|max:100',
                'category' => 'nullable|string|max:100',
                'ration_card_number' => 'nullable|string|max:50',
                'voter_image' => 'nullable|image|mimes:jpg,jpeg,png,gif|max:30720', // 30MB max
            ]);

            // Step 2: Prepare voter data from form input
            $voterData = [
                'voter_id_number' => $request->voter_id_number,
                'name' => $request->name,
                'gender' => $request->gender,
                'year_of_birth' => $request->year_of_birth,
                'date_of_birth' => $request->date_of_birth,
                'anniversary_date' => $request->anniversary_date,
                'is_head' => $request->boolean('is_head', false),
                'booth_id' => $request->booth_id,
                'booth_number' => $request->booth_number,
                'mobile_number' => $request->mobile_number,
                'aadhar_number' => $request->aadhar_number,
                'street_id' => $request->street_id,
                'street_name' => $request->street_name,
                'ward_no' => $request->ward_no,
                'caste' => $request->caste,
                'category' => $request->category,
                'ration_card_number' => $request->ration_card_number,
            ];

            // Step 3: Handle image upload if present
            $imagePath = null;
            if ($request->hasFile('voter_image')) {
                $imagePath = $request->file('voter_image')->store('voter_uploads', 'public');
                $voterData['voter_image'] = $imagePath;
            }

            // Step 4: Create voter using the service
            $result = $this->voterService->createVoter($voterData);
            
            // Step 5: If voter creation failed and we uploaded an image, clean it up
            if ($result->getStatusCode() !== 200 && $imagePath) {
                Storage::disk('public')->delete($imagePath);
            }

            return $result;
        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->response->respondWithError('Validation failed: ' . implode(', ', $e->validator->errors()->all()), 400);
        } catch (\Exception $e) {
            // Clean up uploaded image if something went wrong
            if (isset($imagePath) && $imagePath) {
                Storage::disk('public')->delete($imagePath);
            }
            
            return $this->response->respondWithError('Failed to create voter: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Import voters from CSV file
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function importVoters(Request $request)
    {
        // CSV import endpoint removed. Use Excel import (`importVotersExcel`) or other supported flows.
        return $this->response->respondWithError('CSV import endpoint has been removed', 410);
    }

    /**
     * Import voters from Excel file
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function importVotersExcel(Request $request)
    {
        try {
            // Validate the uploaded file
            $request->validate([
                'excel_file' => 'required|file|mimes:xlsx,xls|max:10240', // 10MB max
            ]);

            if (!$request->hasFile('excel_file')) {
                return $this->response->respondWithError('No Excel file uploaded', 400);
            }

            $file = $request->file('excel_file');
            $filePath = $file->getPathname();

            // Process the Excel file
            return $this->voterService->importVotersFromExcel($filePath);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->response->respondWithError('Validation failed: ' . implode(', ', $e->validator->errors()->all()), 400);
        } catch (\Exception $e) {
            return $this->response->respondWithError('Failed to import Excel voters: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Download CSV template for voter import
     * 
     * @return \Illuminate\Http\Response
     */
    public function downloadTemplate()
    {
        return $this->voterService->generateCsvTemplate();
    }

    /**
     * Download Excel template for voter import
     * 
     * @return \Illuminate\Http\Response
     */
    public function downloadExcelTemplate()
    {
        return $this->voterService->generateExcelTemplate();
    }

    /**
     * Helper method to format voter data consistently
     * 
     * @param Voter $voter
     * @return array
     */
    private function formatVoterData($voter)
    {
        return [
            'id' => $voter->id,
            'voter_id_number' => $voter->voter_id_number,
            'name' => $voter->name,
            'gender' => $voter->gender,
            'year_of_birth' => $voter->year_of_birth,
            'age' => date('Y') - $voter->year_of_birth,
            'is_head' => $voter->is_head,
            'booth_id' => $voter->booth_id,
            'booth_number' => $voter->booth_number,
            'mobile_number' => $voter->mobile_number,
            'aadhar_number' => $voter->aadhar_number,
            'street_id' => $voter->street_id,
            'street_name' => $voter->street_name ?? $voter->street->street_name ?? null,
            'ward_no' => $voter->ward_no,
            'caste' => $voter->caste,
            'category' => $voter->category,
            'ration_card_id' => $voter->ration_card_id,
            'ration_card_number' => $voter->rationCard->ration_card_number ?? null,
            'address' => $voter->rationCard->address ?? null,
            'voter_image' => $voter->voter_image ? asset('storage/' . $voter->voter_image) : null,
        ];
    }

    /**
     * Map existing voter to ration card
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function mapVoterToRationCard(Request $request)
    {
        $data = $request->getContent();

        if (empty($data)) {
            return $this->response->respondWithError('Request data is empty');
        }

        // Decrypt the incoming data
        $decryptedData = DecryptUtils::decryptRequestData($data);

        if ($decryptedData == null || !is_array($decryptedData)) {
            return $this->response->respondWithError('Invalid or empty request data');
        }

        // Pass to service
        return $this->voterService->mapVoterToRationCard($decryptedData);
    }

    /**
     * Map existing voter to ration card (multipart/form-data version)
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function mapVoterToRationCardForm(Request $request)
    {
        try {
            // Validate the form data
            $request->validate([
                'voter_id_number' => 'required|string|max:50',
                'ration_card_number' => 'required|string|max:50',
            ]);

            // Prepare data array
            $data = [
                'voter_id_number' => $request->voter_id_number,
                'ration_card_number' => $request->ration_card_number,
            ];

            // Pass to service
            return $this->voterService->mapVoterToRationCard($data);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return $this->response->respondWithError('Validation failed: ' . implode(', ', $e->validator->errors()->all()), 400);
        } catch (\Exception $e) {
            return $this->response->respondWithError('Failed to map voter to ration card: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Search voter by voter ID number (for testing/verification)
     * 
     * @param string $voterIdNumber
     * @return \Illuminate\Http\JsonResponse
     */
    public function searchVoterByIdNumber($voterIdNumber)
    {
        try {
            $voter = \App\Models\Voter::where('voter_id_number', $voterIdNumber)
                                    ->where('is_deleted', false)
                                    ->with(['rationCard', 'street', 'booth'])
                                    ->first();

            if (!$voter) {
                return $this->response->respondWithError("Voter not found with ID number: {$voterIdNumber}", 404);
            }

            $formattedVoter = $this->formatVoterData($voter);

            return $this->response->respondWithData('Voter found successfully', $formattedVoter, new \Illuminate\Http\Response);

        } catch (\Exception $e) {
            return $this->response->respondWithError('Error searching for voter: ' . $e->getMessage(), 500);
        }
    }
}
