<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Repositories\RationCardRepository;
use Illuminate\Support\Facades\Storage;

class ImageUploadController extends Controller
{
    protected $rationCardRepository;

    public function __construct(RationCardRepository $rationCardRepository)
    {
        $this->rationCardRepository = $rationCardRepository;
    }

    public function uploadImage(Request $request)
    {
        // Validate incoming request
        $request->validate([
            'image' => 'required|image|mimes:jpg,jpeg,png,gif|max:500',
            'ration_card_id' => 'required|integer|exists:ration_cards,id', // ✅ link to ration card
        ]);

        if ($request->hasFile('image')) {
            // Store image in "public/uploads" folder
            $path = $request->file('image')->store('uploads', 'public');

            // Fetch the ration card
            $rationCard = $this->rationCardRepository->getById($request->ration_card_id);

            if (!$rationCard) {
                return response()->json(['message' => 'Ration card not found.'], 404);
            }

            // Save the image path in the ration_cards table
            $rationCard->head_photo = $path;
            $rationCard->save();

            // Return success response
            return response()->json([
                'message' => 'Image uploaded and path saved successfully!',
                'path' => $path,
                'ration_card' => $rationCard
            ], 200);
        }

        return response()->json(['message' => 'No image file uploaded.'], 400);
    }

 public function deleteImage(Request $request)
    {
        // Validate incoming request
        $request->validate([
            'ration_card_id' => 'required|integer|exists:ration_cards,id', // Ration card ID
        ]);

        // Fetch the ration card
        $rationCard = $this->rationCardRepository->getById($request->ration_card_id);

        if (!$rationCard) {
            return response()->json(['message' => 'Ration card not found.'], 404);
        }

        // Check if the ration card has an image
        if (!$rationCard->head_photo) {
            return response()->json(['message' => 'No image found for this ration card.'], 400);
        }

        // Delete the image from storage
        $imagePath = $rationCard->head_photo;
        if (Storage::disk('public')->exists($imagePath)) {
            Storage::disk('public')->delete($imagePath);
        }

        // Remove the image path from the database
        $rationCard->head_photo = null;
        $rationCard->save();

        // Return success response
        return response()->json([
            'message' => 'Image deleted successfully!',
        ], 200);
    }

    /**
     * Upload voter image
     */
    public function uploadVoterImage(Request $request)
    {
        // Validate incoming request
        $request->validate([
            'image' => 'required|image|mimes:jpg,jpeg,png,gif|max:30720', // 30MB max
            'voter_id' => 'required|integer|exists:voters,id',
        ]);

        if ($request->hasFile('image')) {
            // Store image in "public/voter_uploads" folder
            $path = $request->file('image')->store('voter_uploads', 'public');

            // Fetch the voter
            $voter = \App\Models\Voter::find($request->voter_id);

            if (!$voter) {
                return response()->json(['message' => 'Voter not found.'], 404);
            }

            // Delete old image if exists
            if ($voter->voter_image && Storage::disk('public')->exists($voter->voter_image)) {
                Storage::disk('public')->delete($voter->voter_image);
            }

            // Save the new image path in the voters table
            $voter->voter_image = $path;
            $voter->save();

            // Return success response
            return response()->json([
                'message' => 'Voter image uploaded successfully!',
                'path' => $path,
                'voter' => $voter
            ], 200);
        }

        return response()->json(['message' => 'No image file uploaded.'], 400);
    }

    /**
     * Delete voter image
     */
    public function deleteVoterImage(Request $request)
    {
        // Validate incoming request
        $request->validate([
            'voter_id' => 'required|integer|exists:voters,id',
        ]);

        // Fetch the voter
        $voter = \App\Models\Voter::find($request->voter_id);

        if (!$voter) {
            return response()->json(['message' => 'Voter not found.'], 404);
        }

        // Check if the voter has an image
        if (!$voter->voter_image) {
            return response()->json(['message' => 'No image found for this voter.'], 400);
        }

        // Delete the image from storage
        $imagePath = $voter->voter_image;
        if (Storage::disk('public')->exists($imagePath)) {
            Storage::disk('public')->delete($imagePath);
        }

        // Remove the image path from the database
        $voter->voter_image = null;
        $voter->save();

        // Return success response
        return response()->json([
            'message' => 'Voter image deleted successfully!',
        ], 200);
    }
}
