# Voter Bulk Import Documentation

## Overview
The bulk import functionality allows you to upload a CSV file containing multiple voters' data to import them into the database in a single operation.

## Features
✅ **CSV File Import** - Import voters from CSV files
✅ **Flexible Column Mapping** - Supports various column header formats
✅ **Data Validation** - Validates each row before import
✅ **Error Reporting** - Detailed error messages for failed imports
✅ **Duplicate Detection** - Identifies existing voter IDs
✅ **Template Generation** - Download sample CSV template
✅ **Batch Processing** - Handles large files efficiently

---

## API Endpoints

### 1. Import Voters from CSV
**Endpoint:** `POST /api/voters/import`
**Content-Type:** `multipart/form-data`

#### Request Parameters:
- `csv_file` (required) - CSV file (max 10MB)

#### Example using cURL:
```bash
curl -X POST "http://localhost/api/voters/import" \
  -F "csv_file=@voters_data.csv"
```

#### Example using JavaScript:
```javascript
const formData = new FormData();
formData.append('csv_file', csvFile); // File object from input

fetch('/api/voters/import', {
  method: 'POST',
  body: formData
})
.then(response => response.json())
.then(data => console.log(data));
```

#### Example using HTML Form:
```html
<form action="/api/voters/import" method="POST" enctype="multipart/form-data">
  <input type="file" name="csv_file" accept=".csv" required>
  <button type="submit">Import Voters</button>
</form>
```

#### Success Response:
```json
{
  "status": 200,
  "message": "Import completed. 150 voters imported successfully, 5 errors occurred",
  "data": {
    "success_count": 150,
    "error_count": 5,
    "total_processed": 155,
    "errors": [
      "Row 3: Mobile Number must be 10 digits.",
      "Row 7: Voter ID 'BCC123456' already exists",
      "Row 12: Name is required"
    ]
  }
}
```

### 2. Download CSV Template
**Endpoint:** `GET /api/voters/download-template`

#### Example:
```bash
curl -X GET "http://localhost/api/voters/download-template" \
  -o voter_template.csv
```

This downloads a sample CSV file with proper headers and example data.

---

## CSV File Format

### Required Headers (case-insensitive):
The CSV file must include at least these columns:
- `voter_id_number` (required)
- `name` (required)  
- `gender` (required)

### All Supported Headers:
```csv
voter_id_number,name,gender,year_of_birth,date_of_birth,anniversary_date,is_head,booth_id,booth_number,mobile_number,aadhar_number,street_id,street_name,ration_card_number
```

### Alternative Header Names:
The system recognizes various column header formats:

| Field | Accepted Header Names |
|-------|----------------------|
| `voter_id_number` | voter_id_number, voter id number, voter_id, id_number, voter id |
| `name` | name, voter_name, full_name, voter name |
| `gender` | gender, sex |
| `year_of_birth` | year_of_birth, birth_year, yob, year of birth |
| `date_of_birth` | date_of_birth, dob, birth_date, date of birth |
| `anniversary_date` | anniversary_date, anniversary, marriage_date |
| `is_head` | is_head, head_of_family, family_head, head |
| `booth_id` | booth_id, booth id |
| `booth_number` | booth_number, booth number, booth_no, booth |
| `mobile_number` | mobile_number, mobile, phone, contact, mobile number |
| `aadhar_number` | aadhar_number, aadhar, aadhaar, aadhar number |
| `street_id` | street_id, street id |
| `street_name` | street_name, street, address, street name |
| `ration_card_number` | ration_card_number, ration card, ration_card, card_number |

### Sample CSV Content:
```csv
voter_id_number,name,gender,year_of_birth,date_of_birth,anniversary_date,is_head,booth_id,booth_number,mobile_number,aadhar_number,street_id,street_name,ration_card_number
BCC1234567,RAJESWARI,Male,1989,1989-06-09,2017-06-26,0,18,B003,9698979194,348234234234,10,"Mettu street, vikravandi",
BCC1234568,KUMAR,Male,1985,1985-03-15,,1,19,B004,9876543210,123456789012,11,Main street,RC123456
BCC1234569,PRIYA,Female,1992,1992-12-25,2020-02-14,0,18,B003,8765432109,987654321098,10,"Mettu street, vikravandi",
```

---

## Data Validation Rules

### Required Fields:
- `voter_id_number` - Must be unique
- `name` - Cannot be empty
- `gender` - Must be Male, Female, or Other

### Optional Field Validation:
- `mobile_number` - Must be exactly 10 digits
- `aadhar_number` - Must be exactly 12 digits  
- `year_of_birth` - Must be between 1900 and current year
- `date_of_birth` - Must be valid date in YYYY-MM-DD format
- `anniversary_date` - Must be valid date in YYYY-MM-DD format
- `ration_card_number` - Must exist in ration_cards table (if provided)

### Gender Normalization:
- `M`, `Male`, `man` → Male
- `F`, `Female`, `woman` → Female
- Other values → Other

### Boolean Fields (`is_head`):
- `1`, `true`, `yes`, `y` → true
- All other values → false

### Date Format Support:
The system accepts dates in multiple formats:
- `YYYY-MM-DD` (preferred)
- `DD/MM/YYYY`
- `DD-MM-YYYY`
- `MM/DD/YYYY`
- `MM-DD-YYYY`
- `YYYY/MM/DD`

---

## Error Handling

### Common Errors:
1. **File Issues:**
   - "No CSV file uploaded"
   - "CSV file is empty or invalid"
   - File size exceeds 10MB limit

2. **Data Issues:**
   - "Row X: Voter ID Number is required"
   - "Row X: Name is required"
   - "Row X: Mobile Number must be 10 digits"
   - "Row X: Aadhar Number must be 12 digits"

3. **Duplicate Issues:**
   - "Row X: Voter ID 'BCC123456' already exists"
   - "Row X: Ration card number 'RC123456' does not exist"

### Error Response Format:
```json
{
  "status": 200,
  "message": "Import completed. 8 voters imported successfully, 2 errors occurred",
  "data": {
    "success_count": 8,
    "error_count": 2,
    "total_processed": 10,
    "errors": [
      "Row 3: Mobile Number must be 10 digits.",
      "Row 7: Voter ID 'BCC999999' already exists"
    ]
  }
}
```

---

## Best Practices

### 1. File Preparation:
- Use UTF-8 encoding for CSV files
- Ensure consistent date formats
- Remove empty rows
- Validate data before upload

### 2. Large Files:
- For files with 1000+ records, consider splitting into smaller batches
- Monitor the import progress through error messages
- Process during off-peak hours

### 3. Error Resolution:
- Fix errors in the original CSV file
- Re-run import for failed rows only
- Check duplicate voter IDs before import

### 4. Data Quality:
- Use the template file as a starting point
- Validate mobile numbers and Aadhar numbers
- Ensure booth_id and street_id exist in your database

---

## Complete Workflow Example

### Step 1: Download Template
```bash
curl -O http://localhost/api/voters/download-template
```

### Step 2: Prepare Your Data
Fill the template with your voter data, following the validation rules.

### Step 3: Upload and Import
```bash
curl -X POST "http://localhost/api/voters/import" \
  -F "csv_file=@my_voters.csv"
```

### Step 4: Review Results
Check the response for success/error counts and fix any issues.

---

## Integration Tips

### Frontend Integration:
```javascript
// File upload with progress tracking
async function importVoters(file) {
  const formData = new FormData();
  formData.append('csv_file', file);
  
  try {
    const response = await fetch('/api/voters/import', {
      method: 'POST',
      body: formData
    });
    
    const result = await response.json();
    
    if (result.status === 200) {
      console.log(`✅ ${result.data.success_count} voters imported`);
      if (result.data.error_count > 0) {
        console.log(`⚠️ ${result.data.error_count} errors:`, result.data.errors);
      }
    }
  } catch (error) {
    console.error('Import failed:', error);
  }
}
```

### Error Handling:
```javascript
function displayImportResults(result) {
  const { success_count, error_count, errors } = result.data;
  
  // Show success message
  if (success_count > 0) {
    showSuccessMessage(`${success_count} voters imported successfully`);
  }
  
  // Show errors
  if (error_count > 0) {
    errors.forEach(error => showErrorMessage(error));
  }
}
```

This bulk import feature significantly improves efficiency for organizations managing large voter databases!